"""
Module contains agent-example websocket handler.

The tasks of this module:

    - to provide websocket subscription possibility

    - filter requests by comparison of specified query parameters and available streams

    - log errors

"""

import asyncio

from aiohttp import web
from websockets import ConnectionClosed


async def ws_handler(request):
    """
    Handler intended for processing websocket requests:
    - if required stream not exists or doesn't match account id & analytics index - raise an exception
    - otherwise, add subscriber to ws provider and send an event every second to subscriber
    - finish when subscriber close connection or stream removed from ws provider (due to it is done or failed)
    """
    ws = web.WebSocketResponse()

    wsProvider = request.app.agent.wsProvider
    streamId = request.query.get("stream_id")
    accountId = request.query.get("account_id")
    analyticIndex = int(request.query.get("analytic_index"))

    if not wsProvider.contains(streamId=streamId, accountId=accountId, analyticIndex=analyticIndex):
        raise ValueError(f"Stream with id {streamId} not found")
    await ws.prepare(request=request)
    wsProvider.addSubscriber(streamId=streamId, wsConnection=ws, analyticIndex=analyticIndex)
    try:
        while True:
            await asyncio.sleep(1)
            if not wsProvider.contains(streamId=streamId, analyticIndex=analyticIndex):
                break
    except ConnectionClosed as e:
        print(f"WS connection closed with code {e.code}: {e.reason}")
    except asyncio.CancelledError:
        print("Client disconnected session")
    except Exception:
        print("keepalive ping failed", exc_info=True)
    finally:
        wsProvider.removeSubscriber(streamId=streamId, wsConnection=ws, analyticIndex=analyticIndex)

    return ws
