"""
Module contains agent-example high-level, initialize and shutdown methods


The tasks of this module:

    - provide function that will `register analytics <#anchor-analytics-registration>`_

    - provide function that will `register agent <#anchor-agent-registration>`_

    - provide function that will periodically `get streams for processing <#anchor-get-streams>`_ taking
      into account parameters received in reply of agent registration

    - provide function that will add new stream for processing

    - provide function that `gracefully shutdown video-agent <#anchor-shutdown>`_

"""

import asyncio
import time

import aiohttp
from stream_processing import processStream
from structs_n_consts import ACCOUNT_ID, VIDEO_MANAGER_API_URL
from ws_provider import WSProvider


class Agent:
    """Agent-example main class"""

    # list of analytics which agent-example supports
    myAnalytics: list[str] = ["fake_analytics"]

    def __init__(self, port: int):
        """Initialize agent variables for future usage"""
        self.port = port
        self.session: aiohttp.ClientSession | None = None
        self.agentId: str | None = None
        self.refreshPeriod: int | None = None
        self.alivePeriod: int | None = None
        self.tasks: dict[str, asyncio.Task] = {}
        self.wsProvider: WSProvider = WSProvider()

    async def makeRequest(self, uri: str, method: str = "POST", raiseForStatus: bool = True, json: dict | None = None):
        """Make HTTP-request with minimalistic wrap using 1 aiohttp session which is already initialized"""
        return await getattr(self.session, method.lower())(
            url=f"{VIDEO_MANAGER_API_URL}{uri}",
            json=json,
            headers={"Luna-Account-Id": ACCOUNT_ID},
            raise_for_status=raiseForStatus,
        )

    async def registerMyAnalytics(self):
        """Register analytics which agent-example supports"""
        for analyticsName in self.myAnalytics:
            reply = await self.makeRequest(
                uri="/analytics", json={"analytic_name": analyticsName}, raiseForStatus=False
            )
            if reply.status > 300 and reply.status != 409:
                replyJson = await reply.json()
                raise ValueError("Failed to register analytic" + str(replyJson))
        print("Success: agent analytics registration")

    async def registerAgent(self):
        """Register agent itself"""
        reply = await self.makeRequest(
            uri="/agents",
            json={
                "name": "my-agent",
                "max_stream_count": 2,
                "port": self.port,
                "api_version": 2,
                "analytic_names": self.myAnalytics,
            },
            raiseForStatus=False,
        )
        replyJson = await reply.json()
        if reply.status >= 300:
            raise ValueError("Failed to register agent: " + str(replyJson))
        self.agentId = replyJson["agent_id"]
        self.refreshPeriod = replyJson["refresh_period"]
        self.alivePeriod = replyJson["alive_period"]
        print("Success: agent registration")

    async def run(self):
        """
        Main agent execution point:
        - initialize aiohttp session for future usage
        - register analytics, agent
        - execute main stream processing cycle
        """
        self.session = aiohttp.ClientSession()
        await self.registerMyAnalytics()
        await self.registerAgent()
        replyFailTime: float | None = None

        while True:
            if replyFailTime is not None and time.time() - replyFailTime >= self.alivePeriod:
                print(f"Fail to connect manager for {self.alivePeriod} seconds, shut down")
                break
            await asyncio.sleep(self.refreshPeriod)
            reply = await self.makeRequest(uri=f"/agents/{self.agentId}/streams", method="POST", raiseForStatus=False)
            replyJson = await reply.json()
            if replyFailTime is None and reply.status >= 300:
                replyFailTime = time.time()
            else:
                replyFailTime = None

            for streamAsDict in replyJson["streams"]:
                self.tasks[streamAsDict["stream_id"]] = asyncio.create_task(
                    processStream(
                        streamAsDict=streamAsDict,
                        wsProvider=self.wsProvider,
                        agentId=self.agentId,
                        session=self.session,
                    )
                )

    async def runAsTask(self, app):
        """Execute agent main cycle as asyncio-task"""
        asyncio.create_task(self.run())

    async def close(self, app):
        """
        Shutdown agent:
        - Stop all active stream processing tasks
        - Close all ws connections
        - Remove agent from video-manager making http-request
        - Close aiohttp session
        """
        for activeTask in self.tasks.values():
            if not activeTask.cancelled() and not activeTask.done():
                activeTask.cancel()

        await self.wsProvider.close()
        if self.agentId is not None:
            await self.makeRequest(uri=f"/agents/{self.agentId}", method="delete")
        if self.session is not None:
            await self.session.close()
        print("Agent has been stopped")
