"""
Module contains agent-example feedback processing functions.

The tasks of this module:

    - provide function which `periodically send feedback <#anchor-feedback>`_ to Luna-Video-Manager

"""

import asyncio
import datetime
import time

import aiohttp
from structs_n_consts import VIDEO_MANAGER_API_URL, StreamStatus


class Feedback:
    """
    Primitive feedback sender which implements only basic login
    """

    def __init__(self, streamAsDict: dict, session: aiohttp.ClientSession, agentId: str):
        """Initialize internal variables for future usage"""
        self.streamAsDict = streamAsDict
        self.session = session
        self.status = StreamStatus.in_progress
        self.agentId = agentId

    async def _send(self):
        """
        Send feedback to video-manager, log, return reply from video-manager
        Warnings:
            - is valid for video-manager
            - is sending useless information as `error`
            - fake `time` value (last processed frame time)
            - `[0]` analytics index, which is not applicable for splittable streams or streams with > 1 analytics
        """
        reply = await self.session.post(
            url=f"{VIDEO_MANAGER_API_URL}/agents/{self.agentId}/streams/feedback",
            json={
                "streams": [
                    {
                        "stream_id": self.streamAsDict["stream_id"],
                        "status": self.status.value,
                        "error": "error" if self.status == StreamStatus.failed else "no error",
                        "version": self.streamAsDict["version"],
                        "time": str(datetime.datetime.now(datetime.UTC).replace(tzinfo=None)),
                        "last_processed_frame_time": float(time.time()),
                        "analytic_indexes": [0],
                    }
                ]
            },
        )
        print("feedback has been sent")
        return reply

    async def run(self):
        """
        Endless feedback which works every `feedback_frequency`(received from video-manager)
        sending cycle which sends feedback until (one of):
        - video-manager reply with `stop` signal for current stream (feedback with confirmation also must be sent)
        - stream is done or failed what is due to agent stream processing algorythm
        """
        while True:
            reply = await self._send()
            replyJson = await reply.json()
            if replyJson["streams"] and replyJson["streams"][0]["action"] == "stop":
                self.status = StreamStatus.stop
                await self._send()
                break
            if self.status in (StreamStatus.done, StreamStatus.failed):
                break
            await asyncio.sleep(self.streamAsDict["feedback_frequency"])
