"""
Module contains agent-example stream processing functions.

The tasks of this module:

    - covers the entire stream process from start to finish

    - log (print in the example) stream processing start, error, finish, callback sending, etc

    - launch processing in a coordinated manner taking into account analytics working intervals

    - send decoded frames to analytics and process results

"""

import asyncio

import aiohttp
from analytics import FakeAnalytics
from decoder import Decoder
from feedback import Feedback
from intervals_solver import isAnalyticsMustWorks
from structs_n_consts import StreamStatus
from ws_provider import WSProvider


async def processStream(streamAsDict: dict, session: aiohttp.ClientSession, agentId: str, wsProvider: WSProvider):
    """
    Process stream:
    - Create fake decoder and analytics; feedback sender and add stream to ws provider
    - while decoder return frames, check whether analytics must be executed due to its working intervals
    - if analytics must not work send corresponding feedback and log
    - if analytics must work fake `execute callbacks`, log and send events to all ws subscribers
    - if any exception occurred, fail stream and remove it from ws provider
    - if stream processing is done (no more frames from decoder), finish stream processing with corresponding feedback
    """
    print(f"Start processing stream: {streamAsDict['stream_id']}")
    decoder = Decoder(url=streamAsDict["data"]["reference"], **streamAsDict["data"])
    analyticsManager = FakeAnalytics(streamAsDict["analytics"])
    feedbackSender = Feedback(streamAsDict=streamAsDict, session=session, agentId=agentId)
    feedbackTask = asyncio.create_task(feedbackSender.run())
    wsProvider.addStream(stream=streamAsDict)

    try:
        async for frame in decoder.nextFrame():
            if feedbackTask.cancelled() or feedbackTask.done():
                print("Stream processing has been stopped due to manager reply")
                return
            if not isAnalyticsMustWorks(intervals=streamAsDict["analytics"][0]["intervals"]["intervals"]):
                feedbackSender.status = StreamStatus.paused
                analyticsManager.enabled = False
                print("Analytics processing is paused due to its scheduling")
                continue
            else:
                analyticsManager.enabled = True
                feedbackSender.status = StreamStatus.in_progress
                await asyncio.sleep(1)
            async for event in analyticsManager.events(frame):
                wsProvider.wsBroadcast(stream_id=streamAsDict["stream_id"], analyticsResults=event)
                print(f"Callbacks for event have been sent! Event: {event}")

    except Exception as exc:
        print(f"Exception occurred during stream processing: {str(exc)}")
        feedbackSender.status = StreamStatus.failed
        wsProvider.removeStream(streamId=streamAsDict["stream_id"])

    else:
        print(f"Finish processing stream: {streamAsDict['stream_id']}")
        feedbackSender.status = StreamStatus.done
        wsProvider.finishStream(streamId=streamAsDict["stream_id"])
